/*****************************************************************************
| FILE:         inc_fi_main.cpp
| PROJECT:      G3g
| SW-COMPONENT: ---
|------------------------------------------------------------------------------
| DESCRIPTION:  Message dispatcher instance. Will provide all required functionality
|
|------------------------------------------------------------------------------
| COPYRIGHT:    (c) 2013 Robert Bosch GmbH
| HISTORY:
| Date      | Modification               | Author
| 08.06.13  | Initial revision           | Ranjit Katuri
| --.--.--  | ----------------           | -------, -----
|
|*****************************************************************************/

#include "inc_fi_main.h"
#include "inc_fi_handler.h"

#include <iostream>

DLT_IMPORT_CONTEXT(incfi_dltlog)

using namespace std;

inc_fi_main* inc_fi_main::m_ptrself = NULL;


/**
* Constructor
*/
inc_fi_main::inc_fi_main()
:m_dispatchtable()
{
  //Initialize mutex
  pthread_mutex_init(&m_dpmutex,NULL);
}

/**
* Destructor
*/
inc_fi_main::~inc_fi_main()
{
  m_dispatchtable.clear();
  //optional pthread_mutex_destroy
}

/**
* Singleton instance creator
*/
inc_fi_main* inc_fi_main::poGetInstance()
{
  if(m_ptrself == NULL)
  {
    m_ptrself = new inc_fi_main();
  }
  return m_ptrself;
}

/**
* Message dispatcher function
*/
void inc_fi_main::vOnNewMessage (amt_tclBaseMessage* pomsg)
{
  if(pomsg != NULL)
  {
    amt_tclServiceData oServiceData(pomsg);
    uint16_t u16ServiceID = oServiceData.u16GetServiceID();
    //Check if locking is needed here
    pthread_mutex_lock(&m_dpmutex);
    //Known issue , what happens when deletion happens in between
    map<uint16_t, inc_fi_handler*>::iterator it = m_dispatchtable.find(u16ServiceID);
    pthread_mutex_unlock(&m_dpmutex);
    if(it != m_dispatchtable.end())
    {
      if(it->second != NULL)
      {
        //Dispatch message
        it->second->vDispatchMessage(&oServiceData);
      }
    }
    else
    {
      DLT_LOG(incfi_dltlog,DLT_LOG_FATAL,DLT_STRING("inc_fi_main::vOnNewMessage Message for unregisted handler !!! srvid="),DLT_UINT(u16ServiceID));
    }
  }
}

/**
* Function for Registering a inc fi handler
*/
bool  inc_fi_main::bRegisterHandler(uint16_t srvid, inc_fi_handler* ptrhandler)
{
  bool ret = false;

  DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_main::bRegisterHandler REG handler : srvid="),DLT_UINT(srvid));

  if(ptrhandler != NULL)//Null ptr validation
  {
    pthread_mutex_lock(&m_dpmutex);//Lock it
    //Known issue , what happens when deletion happens after I release the lock??
    map<uint16_t, inc_fi_handler*>::iterator it = m_dispatchtable.find(srvid);
    if(it == m_dispatchtable.end())
    {
      ret = true;
      m_dispatchtable[srvid]= ptrhandler;
      //Invoke register method of the service
      if(!ptrhandler->bRegisterService())
      {
        DLT_LOG(incfi_dltlog,DLT_LOG_FATAL,DLT_STRING("inc_fi_main::bRegisterHandler Failed to send service register !!! : srvid="),DLT_UINT(srvid));
      }
    }
    else
    {
      DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_main::bRegisterHandler Handler already registered !!! : srvid="),DLT_UINT(srvid));
    }
    pthread_mutex_unlock(&m_dpmutex);//Unlock it
  }
  return ret;
}
/**
* Function to remove a handler
*/
bool  inc_fi_main::bUnRegisterHandler(uint16_t srvid)
{
  DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_main::bUnRegisterHandler UNREG handler : srvid="),DLT_UINT(srvid));
  //ToDO: Do a safe removal... integrate this to main loop
  pthread_mutex_lock(&m_dpmutex);//Lock it
  //Invoke a service unregister
  map<uint16_t, inc_fi_handler*>::iterator it = m_dispatchtable.find(srvid);
  if(it != m_dispatchtable.end())
  {
    if(it->second != NULL)
    {
      it->second->bUnRegisterService();
    }
  }
  else
  {
    DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_main::bUnRegisterHandler No such handler : srvid="),DLT_UINT(srvid));
  }

  m_dispatchtable.erase(srvid);
  pthread_mutex_unlock(&m_dpmutex);//Unlock it
  return true;
}

/**
* Function to handle the service state
*/
void inc_fi_main::vOnServiceState
    (
      uint16_t u16ServiceId,
      uint16_t u16ServerId,
      uint16_t u16RegisterId,
      uint8_t u8ServiceState,
      uint16_t u16SubId  )
{
  DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_main::vOnServiceState : srvid="),DLT_UINT(u16ServiceId),\
    DLT_STRING("ServerID:"),DLT_UINT(u16ServerId),DLT_STRING("Regid:"),DLT_UINT(u16RegisterId),DLT_STRING("State:"),DLT_UINT(u8ServiceState));
  pthread_mutex_lock(&m_dpmutex);//Lock it

  map<uint16_t, inc_fi_handler*>::iterator it = m_dispatchtable.find(u16ServiceId);
  if(it != m_dispatchtable.end())
  {
    if(it->second != NULL)
    {
      if(it->second->m_u16appid == u16ServerId)
      {
        //Invoke set service state
        it->second->vSetServiceState(u16RegisterId,u8ServiceState,u16SubId);
      }
    }
  }
  else
  {
    DLT_LOG(incfi_dltlog,DLT_LOG_INFO,DLT_STRING("inc_fi_main::vOnServiceState No such handler : srvid="),DLT_UINT(u16ServiceId));
  }
  pthread_mutex_unlock(&m_dpmutex);//Unlock it
}

